// DOM element references
const uploadZone = document.getElementById("uploadZone");
const fileInput = document.getElementById("fileInput");
const clearBtn = document.getElementById("clearBtn");
const status = document.getElementById("status");
const foundFile = document.getElementById("foundFile");
const projectType = document.getElementById("projectType");
const preview = document.getElementById("preview");

// Store blob URLs for cleanup
let blobUrls = [];

// Display status message
function showStatus(message, type = "info") {
  status.textContent = message;
  status.className = `status ${type}`;
  status.style.display = "block";
  console.log(`[${type.toUpperCase()}]`, message);
}

// Hide status message
function hideStatus() {
  status.style.display = "none";
}

// Cleanup blob URLs to free memory
function revokeBlobUrls() {
  blobUrls.forEach((url) => URL.revokeObjectURL(url));
  blobUrls = [];
}

// Upload zone click to open file browser
uploadZone.addEventListener("click", () => fileInput.click());

// Drag over upload zone
uploadZone.addEventListener("dragover", (e) => {
  e.preventDefault();
  uploadZone.classList.add("dragover");
});

// Drag leave upload zone
uploadZone.addEventListener("dragleave", () => {
  uploadZone.classList.remove("dragover");
});

// Drop file on zone
uploadZone.addEventListener("drop", (e) => {
  e.preventDefault();
  uploadZone.classList.remove("dragover");
  const file = e.dataTransfer.files[0];
  if (file) handleZipFile(file);
});

// File input change event
fileInput.addEventListener("change", (e) => {
  const file = e.target.files[0];
  if (file) handleZipFile(file);
});

// Clear button to reset preview
clearBtn.addEventListener("click", () => {
  preview.src = "about:blank";
  fileInput.value = "";
  foundFile.textContent = "—";
  projectType.textContent = "—";
  hideStatus();
  revokeBlobUrls();
});

// Process ZIP file and display in iframe
async function handleZipFile(file) {
  if (!file.name.endsWith(".zip")) {
    showStatus("Please upload a .zip file", "error");
    return;
  }

  hideStatus();
  revokeBlobUrls();

  try {
    const zip = new JSZip();
    const contents = await zip.loadAsync(file);

    // Extract all file paths
    const files = Object.keys(contents.files).filter(
      (path) => !contents.files[path].dir
    );

    // Find HTML file
    let htmlFile = files.find((f) => f.toLowerCase().endsWith("index.html"));
    if (!htmlFile) {
      htmlFile = files.find((f) => f.toLowerCase().endsWith(".html"));
    }

    if (!htmlFile) {
      showStatus("No HTML file found in ZIP", "error");
      return;
    }

    foundFile.textContent = htmlFile;
    projectType.textContent = "ZIP Archive";

    // Create blob URLs with MIME types
    const urlMap = {};
    const mimeTypes = {
      html: "text/html",
      css: "text/css",
      js: "application/javascript",
      json: "application/json",
      png: "image/png",
      jpg: "image/jpeg",
      jpeg: "image/jpeg",
      gif: "image/gif",
      svg: "image/svg+xml",
      webp: "image/webp",
      woff: "font/woff",
      woff2: "font/woff2",
      ttf: "font/ttf",
      eot: "application/vnd.ms-fontobject",
    };

    for (const path of files) {
      const file = contents.files[path];
      const blob = await file.async("blob");

      const ext = path.split(".").pop().toLowerCase();
      const mimeType = mimeTypes[ext] || "application/octet-stream";
      const typedBlob = new Blob([blob], { type: mimeType });
      const blobUrl = URL.createObjectURL(typedBlob);
      blobUrls.push(blobUrl);

      // Store with multiple path variations for flexible resolution
      urlMap[path] = blobUrl;
      urlMap[path.toLowerCase()] = blobUrl;
      const fileName = path.split("/").pop();
      if (!urlMap[fileName]) urlMap[fileName] = blobUrl;
      if (!urlMap[fileName.toLowerCase()])
        urlMap[fileName.toLowerCase()] = blobUrl;
    }

    // Read HTML and update resource paths
    let html = await contents.files[htmlFile].async("text");
    const baseDir = htmlFile.includes("/")
      ? htmlFile.substring(0, htmlFile.lastIndexOf("/") + 1)
      : "";

    // Replace src/href attributes
    html = html.replace(
      /(src|href)=["']([^"']+)["']/gi,
      (match, attr, path) => {
        if (
          path.startsWith("http") ||
          path.startsWith("//") ||
          path.startsWith("data:")
        ) {
          return match;
        }

        const cleanPath = path.replace(/^\.\//, "").replace(/^\//, "");
        const fullPath = baseDir + cleanPath;

        const resolved =
          urlMap[fullPath] ||
          urlMap[fullPath.toLowerCase()] ||
          urlMap[cleanPath] ||
          urlMap[cleanPath.toLowerCase()] ||
          urlMap[path.split("/").pop()] ||
          urlMap[path.split("/").pop().toLowerCase()];

        return resolved ? `${attr}="${resolved}"` : match;
      }
    );

    // Replace CSS url() references
    html = html.replace(/url\(['"]?([^'"()]+)['"]?\)/gi, (match, path) => {
      if (
        path.startsWith("http") ||
        path.startsWith("//") ||
        path.startsWith("data:")
      ) {
        return match;
      }

      const cleanPath = path.replace(/^\.\//, "").replace(/^\//, "");
      const fullPath = baseDir + cleanPath;

      const resolved =
        urlMap[fullPath] ||
        urlMap[fullPath.toLowerCase()] ||
        urlMap[cleanPath] ||
        urlMap[cleanPath.toLowerCase()];

      return resolved ? `url('${resolved}')` : match;
    });

    // Display in iframe
    const htmlBlob = new Blob([html], { type: "text/html" });
    const htmlUrl = URL.createObjectURL(htmlBlob);
    blobUrls.push(htmlUrl);

    preview.src = htmlUrl;
    showStatus("Preview loaded successfully!", "success");
  } catch (error) {
    console.error("Error processing ZIP:", error);
    showStatus("Error processing ZIP: " + error.message, "error");
  }
}
